/*
 * File:       send.c
 * Purpose:    Automate the sending of SMTP mail so I don't have to telnet
 *             all the time.
 * Author:     Brett Looney - brettski@myrealbox.com
 * License:    Take it, use it, have fun - if you modify it, please send
 *             me a copy of your changes.
 * 04-Dec-2002:(v1.12)
 *             Fixed problem where we weren't putting a blank line after the
 *             message header, causing attached files to have their first
 *             line munged.
 *             Now, before anyone critiques my code too hard, I did write
 *             this sometime in '97 or '98 before I had *any* idea about
 *             coding anything network based.
 * 31-May-2004:(v1.13)
 *             Added feature to mail either the first 'x' or last 'x' bytes
 *             of the requested file.
 * 02-May-2005:(v1.14)
 *             Now sends current date/time in email header.
 *             Now optionally sends file with MIME encoding.
 *             Optionally deletes target file after successful send.
 * 14-Sep-2005:(v1.15)
 *             Option to close screen automatically after sending with no
 *             errors or to close after 'n' seconds.
 * 15-Oct-2007:(v1.16)
 *             Made the date field RFC2822 compliant in a very lazy way.
 */

#ifdef NETWARE
#include <conio.h>
#include <netinet/in.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netdb.h>
#include <time.h>

#ifndef NETWARE
#include <netinet/in.h>
#endif

#ifndef FALSE
#define FALSE   (0)
#endif
#ifndef TRUE
#define TRUE    (1)
#endif

#define S_MaxLine         (127)
#define S_VersionString   "1.16"

#ifdef NETWARE
NETDB_DEFINE_CONTEXT;
#endif

struct sockaddr_in RemoteSocket;
char Recipient[ S_MaxLine+1 ], OverrideTo[ S_MaxLine+1 ],
     MailHost[ S_MaxLine+1 ],
     Subject[ S_MaxLine+1 ] = "Blank Subject",
     From[ S_MaxLine+1 ] = "test@test",
     OverrideFrom[ S_MaxLine+1 ] = "",
     MessageFile[ S_MaxLine+1 ] = "",
     Domain[ S_MaxLine+1 ] = "",
     DateString[ S_MaxLine+1 ] = "",
     EOL[ S_MaxLine+1 ] = "\r\n";
int Debugging = FALSE, Responses = FALSE, ShowSent = FALSE, Quiet = FALSE,
    NoBody = FALSE, PrintPercent = TRUE, SendAngles = TRUE,
    AddressInHeader = TRUE, FileLimit = FALSE, SendDate = TRUE,
    MIMEEncoding = FALSE, DeleteTargetFile = FALSE, AutoClose = FALSE;
int WaitOnErrorTime = 0;
long FileSize = 0;

/*
 * Module:  CloseSocket
 * Purpose: As there are two things for us to do when we close a socket,
 *          we'll do them both here.
 */

void CloseSocket(int Socket)
{
  shutdown(Socket, 2);
  close(Socket);
} /* End of CloseSocket() */

/*
 * Module:  GetSocket
 * Purpose: Allocate a socket and connect it to the other end. Return -1
 *          if we can't for some reason.
 */

int GetSocket(void)
{
  int Socket = -1;

  if( (Socket = socket(AF_INET, SOCK_STREAM, 0)) == -1 ) {
    if( Debugging )
      printf("GetSocket: Could not allocate socket.\n");
  }
  else {
    if( Debugging )
      printf("GetSocket: Got socket - connecting to remote host.\n");

    if( connect(Socket, (struct sockaddr *)&RemoteSocket,
                sizeof(struct sockaddr_in)) == -1 ) {
      CloseSocket(Socket);
      Socket = -1;

      if( Debugging ) {
        printf("GetSocket: Could not connect to remote host: ");
        if( errno == ECONNREFUSED )
          printf("Connection refused.\n");
        else
          printf("Timeout. (errno=%d)\n", errno);
      }
    }
    else if( Debugging )
      printf("GetSocket: Connected to remote host - continuing.\n");
  }

  return(Socket);
} /* End of GetSocket() */

/*
 * Module:  ReadLine
 * Purpose: Read a line from the remote host and put it in a string.
 *          Modified to only return up to and including a carraige return
 *          unless the buffer is full.
 */

int ReadLine(int Socket, char *String)
{
  static char Remaining[ S_MaxLine*4 ] = "";

  int Bytes, Status = TRUE;
  char *Pointer;

  if( strlen(Remaining) ) {
    strcpy(String, Remaining);
    Remaining[ 0 ] = (char)NULL;
  }
  else
  {
    Bytes = read(Socket, String, S_MaxLine);
    if( Bytes < 0 )
      Status = FALSE;
    else
      String[ Bytes ] = (char)NULL;
  }

  Pointer = strchr(String, '\n');
  if( Pointer != (char *)NULL ) {
    strcpy(Remaining, ++Pointer);
    *Pointer = (char)NULL;
  }

  if( Responses )
    fprintf(stderr, String);

  return(Status);
} /* End of ReadLine() */

/*
 * Module:  SendLine
 * Purpose: Send a line of text to the remote host.
 */

int SendLine(int Socket, char *String)
{
  int Sent, DataLen = strlen(String), Status = TRUE;

  Sent = write(Socket, String, DataLen);
  if( Sent != DataLen ) {
    Status = FALSE;
    if( Debugging )
      printf("SendLine: Sent only %d of %d bytes.\n", Sent, DataLen);
  }

  if( ShowSent )
    fprintf(stderr, String);

  return(Status);
} /* End of SendLine() */

/*
 * Module:  EncodeBlock
 * Purpose: Encode three 8-bit bytes as a four bytes MIME block.
 */
/* MIME translation table as described in RFC1113 */
static const char MIME[]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

unsigned char *MIMEEncode(unsigned char Input[ 3 ], int Length)
{
  static unsigned char Output[ 5 ];

  Output[ 0 ] = MIME[ Input[0] >> 2 ];
  Output[ 1 ] = MIME[ ((Input[0] & 0x03) << 4) | ((Input[1] & 0xf0) >> 4) ];
  Output[ 2 ] = (unsigned char) (Length > 1 ? MIME[ ((Input[1] & 0x0f) << 2) | ((Input[2] & 0xc0) >> 6) ] : '=');
  Output[ 3 ] = (unsigned char) (Length > 2 ? MIME[ Input[2] & 0x3f ] : '=');
  Output[ 4 ] = (unsigned char)NULL;

  return(Output);
} /* End of MIMEEncode() */

/*
 * Module:  Basename
 * Purpose: Return just the filename from a given path.
 */

char *Basename(char *Path)
{
  static char FileName[ S_MaxLine+1 ];
  char *Pointer;

  Pointer = strrchr(Path, '\\');
  if( Pointer == (char *)NULL ) {
    Pointer = strrchr(Path, ':');
    if( Pointer == (char *)NULL ) {
      strcpy(FileName, Path);
    }
    else
      strcpy(FileName, Pointer+1);
  }
  else
    strcpy(FileName, Pointer+1);

  return(FileName);
} /* End of Basename() */

/*
 * Module:  SendMail
 * Purpose: Using all the given parameters, send some mail.
 */

int SendMail(void)
{
  FILE *DataFile;
  int Socket, PlainMessage = TRUE, Percentage, ThisLine, FileLoop = TRUE,
      BlockLength, i, Block;
  long ExpectedLength = 0, AlreadySent = 0, MIMECounter = 0;
  char LineIn[ S_MaxLine+1 ], LineOut[ S_MaxLine+1 ],
       MIMEBoundary[ S_MaxLine+1 ], InputBlock[ S_MaxLine+1 ];
  struct stat FileInfo;

  if( (Socket = GetSocket()) == -1 ) {
    if( !Quiet && !Debugging )
      printf("Socket allocation failed - no message sent.\n");
    return(FALSE);
  }

  if( MIMEEncoding )
    sprintf(MIMEBoundary, "__RandomBoundaryThing%05d__", Socket);

  if( Debugging ) printf("Getting header from mail host\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  sprintf(LineOut, "HELO %s%s", strlen(Domain) ? Domain : "", EOL);
  if( Debugging ) printf("Sending: %s", LineOut);
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Getting domain response from mail host\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  sprintf(LineOut, "MAIL FROM:%s%s%s%s",
          SendAngles ? "<" : "", From, SendAngles ? ">" : "", EOL);
  if( Debugging ) printf("Sending: %s", LineOut);
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Getting FROM response from mail host\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  sprintf(LineOut, "RCPT TO:%s%s%s%s",
          SendAngles ? "<" : "", Recipient, SendAngles ? ">" : "", EOL);
  if( Debugging ) printf("Sending: %s", LineOut);
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Getting RCPT response from mail host\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  sprintf(LineOut, "DATA%s", EOL);
  if( Debugging ) printf("Sending: %s", LineOut);
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Getting DATA response from mail host\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Sending mail header.");
  if( AddressInHeader ) {
    sprintf(LineOut, "To: %s%sFrom: %s%s", OverrideTo, EOL, OverrideFrom, EOL);
    if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
  }
  else
    if( Debugging ) printf("  Suppressing To: and From: headers.");

  if( SendDate ) {
    sprintf(LineOut, "Date: %s%s", DateString, EOL);
    if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
  }

  if( Debugging ) printf("\n");
  sprintf(LineOut, "Subject: %s%s", Subject, EOL);
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( NoBody ) {
    if( Debugging ) printf("No message body to send.\n");
  }
  else {
    if( MIMEEncoding ) {
      sprintf(LineOut, "Mime-Version: 1.0%s", EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

      sprintf(LineOut, "Content-Type: multipart/mixed; boundary=\"%s\"%s",
              MIMEBoundary, EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
    }

    if( Debugging ) printf("Sending end of header marker.\n");
    if( !SendLine(Socket, EOL) ) { CloseSocket(Socket); return(FALSE); }

    if( MIMEEncoding ) {
      if( Debugging ) printf("Sending MIME headers (%s)\n", Basename(MessageFile));

      sprintf(LineOut, "This message is MIME encoded. If you are seeing this your%s", EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
      sprintf(LineOut, "mail client does not support MIME encoding.%s%s", EOL, EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

      sprintf(LineOut, "--%s%s", MIMEBoundary, EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

      sprintf(LineOut, "Content-Disposition: attachment; filename=\"%s\"%s",
              Basename(MessageFile), EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

      sprintf(LineOut, "Content-Transfer-Encoding: base64%s%s", EOL, EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
    }

    if( Debugging ) printf("Sending message text.\n");
    if( strlen(MessageFile) ) {
      DataFile = fopen(MessageFile, "rb");
      if( DataFile == NULL ) {
        if( !Quiet )
          printf("WARNING: %s can't be opened - sending default message.\n",
                 MessageFile);
      }
      else {
        if( !stat(MessageFile, &FileInfo) ) {
          ExpectedLength = FileInfo.st_size;
          if( ExpectedLength == 0 ) {
            PrintPercent = FALSE;
            if( FileLimit && !Quiet )
              printf("WARNING: Cannot determine file size - no file size limit imposed.\n");
            FileLimit = FALSE;
          }
          else if( FileLimit )
            ExpectedLength = labs(FileSize);

          if( FileLimit && (FileSize < 0) ) {
            if( Debugging ) printf("Seeking to EOF minus file limit.\n");
            if( fseek(DataFile, FileInfo.st_size+FileSize, SEEK_SET) ) {
              FileLimit = FALSE;
              fseek(DataFile, 0L, SEEK_SET);
              if( !Quiet )
                printf("WARNING: Seek to middle failed - no file size limit imposed.\n");
            }
          }
        }
        else {
          PrintPercent = FALSE;
          if( FileLimit && !Quiet )
            printf("WARNING: Cannot determine file size - no file size limit imposed.\n");
          FileLimit = FALSE;
        }

        PlainMessage = FALSE;

        while( FileLoop ) {
          if( MIMEEncoding ) {
            ThisLine = 0;
            LineOut[ 0 ] = (char)NULL;

            for(Block = 0; (Block < 18) && FileLoop; Block++) {
              BlockLength = 0;
              for(i = 0; i < 3; i++) {
                InputBlock[ i ] = (unsigned char)getc(DataFile);
                if( feof(DataFile) ) {
                  FileLoop = FALSE;
                  InputBlock[ i ] = (unsigned char)NULL;
                }
                else {
                  BlockLength++;
                  ThisLine++;
                }
              }
              if( BlockLength )
                strcat(LineOut, MIMEEncode(InputBlock, BlockLength));
            }
            MIMECounter += strlen(LineOut);
            strcat(LineOut, EOL);
          }
          else {
            if( fgets(LineOut, S_MaxLine, DataFile) == NULL )
              FileLoop = FALSE;
            ThisLine = strlen(LineOut);
          }

          if( FileLimit && (FileSize > 0) ) {
            if( AlreadySent+ThisLine > labs(FileSize) ) {
              if( Debugging ) printf("File size limit reached - truncating send operation.\n");

              LineOut[ FileSize-AlreadySent ] = (char)NULL;
              FileLoop = FALSE;
            }
          }

          if( !SendLine(Socket, LineOut) ) {
            fclose(DataFile);
            CloseSocket(Socket);
            return(FALSE);
          }

          if( PrintPercent ) {
            AlreadySent += ThisLine;
            Percentage = ((float)AlreadySent/ExpectedLength)*100;
            printf("%d%%\r", Percentage);
          }
        }

        fclose(DataFile);
      }
    }

    if( PlainMessage ) {
      sprintf(LineOut, "<no message text specified>%s", EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
    }

    if( MIMEEncoding ) {
      sprintf(LineOut, "%s--%s--%s", EOL, MIMEBoundary, EOL);
      if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }
    }
  }

  sprintf(LineOut, "%s.%s", EOL, EOL);
  if( Debugging ) printf("Sending end of data marker.\n");
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Getting mail sent response\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  sprintf(LineOut, "QUIT%s", EOL);
  if( Debugging ) printf("Sending: %s", LineOut);
  if( !SendLine(Socket, LineOut) ) { CloseSocket(Socket); return(FALSE); }

  if( Debugging ) printf("Getting quit response\n");
  if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }
  while( LineIn[ 3 ] == '-' )
    if( !ReadLine(Socket, LineIn) ) { CloseSocket(Socket); return(FALSE); }

  CloseSocket(Socket);

  if( !PlainMessage && DeleteTargetFile ) {
    if( Debugging ) printf("Deleting %s\n", MessageFile);
    unlink(MessageFile);
  }

  if( !Quiet )
    if( AlreadySent )
      if( MIMEEncoding )
        printf("Complete (%ld bytes - %ld bytes encoded).\n", AlreadySent, MIMECounter);
      else
        printf("Complete (%ld bytes).\n", AlreadySent);
    else
      printf("Complete.\n");

  return(TRUE);
} /* End of SendMail() */

/*
 * Module:  CheckComms
 * Purpose: Make sure that we can talk via TCP and that everything is set
 *          up properly. Return a TRUE/FALSE value based on that.
 */

int CheckComms(void)
{
  int ReturnValue = TRUE;
  struct hostent *RemoteInfo;

#ifndef NETWARE
  struct servent *ServerInfo;

  ServerInfo = (struct servent *)getservbyname("smtp", "tcp");
  if( ServerInfo == NULL ) {
    ReturnValue = FALSE;
    if( Debugging )
      printf("CheckComms: Could not get server info for smtp\n");
  }
  else {
    if( Debugging )
      printf("CheckComms: Found service information.\n");
#else
  {
#endif
    RemoteInfo = (struct hostent *)gethostbyname(MailHost);
    if( RemoteInfo == NULL ) {
      ReturnValue = FALSE;
      if( Debugging )
        printf("CheckComms: Could not find host info: %s\n", MailHost);
    }
    else {
      if( Debugging )
        printf("CheckComms: Found remote host information. (%d.%d.%d.%d)\n",
               (unsigned char)RemoteInfo->h_addr[ 0 ],
               (unsigned char)RemoteInfo->h_addr[ 1 ],
               (unsigned char)RemoteInfo->h_addr[ 2 ],
               (unsigned char)RemoteInfo->h_addr[ 3 ]);

      memset(&RemoteSocket, (char)NULL, sizeof(struct sockaddr_in));
      RemoteSocket.sin_family = RemoteInfo->h_addrtype;
      memcpy(&RemoteSocket.sin_addr, RemoteInfo->h_addr, RemoteInfo->h_length);
#ifdef NETWARE
      RemoteSocket.sin_port = htons(25);
#else
      RemoteSocket.sin_port = ServerInfo->s_port;
#endif
    }
  }

  return(ReturnValue);
} /* End of CheckComms() */

/*
 * Module:  main
 * Purpose: Read the commandline, and then call the appropriate modules.
 */

void main(int argc, char *argv[])
{
  struct tm *LocalTime;
  time_t SystemTime;
  int PrintHelp = FALSE, i, Errors = FALSE;
  char *MonthName[ 12 ] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
                           "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
  char *DayName[ 7 ] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};

  if( argc < 3 )
    PrintHelp = TRUE;
  else {
    strcpy(Recipient, argv[ 1 ]); strcpy(OverrideTo, argv[ 1 ]);
    strcpy(MailHost, argv[ 2 ]);

    for(i = 3; i < argc; i++) {
      if( argv[ i ][ 0 ] == '+' ) {
        FileSize = atol(&argv[ i ][ 1 ]);
        if( FileSize > 0 )
          FileLimit = TRUE;
      }
      else if( argv[ i ][ 0 ] != '-' )
        PrintHelp = TRUE;
      else {
        switch( argv[ i ][ 1 ] ) {
          case 'a': SendAngles = FALSE; break;
          case 'b': NoBody = TRUE; break;
          case 'c': AddressInHeader = FALSE; break;
          case 'd': strcpy(Domain, &argv[ i ][ 2 ]); break;
          case 'e': ShowSent = TRUE; break;
          case 'f':
            strcpy(From, &argv[ i ][ 2 ]);
            break;
          case 'g': strcpy(OverrideFrom, &argv[ i ][ 2 ]); break;
          case 'i': SendDate = FALSE; break;
          case 'l': strcpy(EOL, "\n"); break;
          case 'm': strcpy(MessageFile, &argv[ i ][ 2 ]); break;
          case 'n': MIMEEncoding = TRUE; break;
          case 'p': PrintPercent = FALSE; break;
          case 'q': Quiet = TRUE; break;
          case 'r': Responses = TRUE; break;
          case 's': strcpy(Subject, &argv[ i ][ 2 ]); break;
          case 't': strcpy(OverrideTo, &argv[ i ][ 2 ]); break;
          case 'x': DeleteTargetFile = TRUE; break;
          case 'y':
            AutoClose = TRUE;
            WaitOnErrorTime = atoi(&argv[ i ][ 2 ]);
            break;
          case 'z': Debugging = TRUE; break;
          case '?':
          default:
            if( isdigit(argv[ i ][ 1 ]) ) {
              FileSize = atol(&argv[ i ][ 1 ])*-1;
              if( FileSize < 0 )
                FileLimit = TRUE;
            }
            else
              PrintHelp = TRUE;
            break;
        }
      }
    }

    if( !strlen(OverrideFrom) )
      strcpy(OverrideFrom, From);
  }

  if( SendDate ) {
    SystemTime = time(NULL);
    LocalTime = localtime(&SystemTime);

    sprintf(DateString, "%s, %d %s %04d %02d:%02d:%02d +0000",
            DayName[ LocalTime->tm_wday ],
            LocalTime->tm_mday,
            MonthName[ LocalTime->tm_mon ],
            LocalTime->tm_year+1900,
            LocalTime->tm_hour,
            LocalTime->tm_min,
            LocalTime->tm_sec);
  }

  if( !PrintHelp ) {
    if( !Quiet ) {
      printf("Mail host: %s\n", MailHost);
      printf("To: %s   From: %s\n", Recipient, From);
      printf("Subject: %s\n", Subject);
      if( SendDate )
        printf("Date: %s\n", DateString);
      if( strcmp(Recipient, OverrideTo) )
        printf("To address in mail header: %s\n", OverrideTo);
      if( strcmp(From, OverrideFrom) )
        printf("From address in mail header: %s\n", OverrideFrom);
      if( strlen(Domain) )
        printf("HELO domain: %s\n", Domain);
      if( strlen(MessageFile) ) {
        printf("Message source: %s", MessageFile);
        if( FileLimit )
          if( FileSize < 0 )
            printf("  (last %ld bytes)", labs(FileSize));
          else
            printf("  (first %ld bytes)", FileSize);
        if( PrintPercent )
          printf("    (with percentage report)");
        if( DeleteTargetFile )
          printf("    (delete after send)");
        printf("\n");
      }
      if( AutoClose ) {
        printf("Screen will be closed automatically if no errors.\n");
        if( WaitOnErrorTime )
          printf("     If errors, screen will be closed after %d seconds.\n", WaitOnErrorTime);
      }
      if( NoBody )
        printf("Message body will be empty.\n");
      else if( MIMEEncoding )
        printf("MIME encoding will be used.\n");
      if( !strcmp(EOL, "\n") )
        printf("WARNING: Sending bare LF - not CR LF.\n");
    }

    if( Debugging || Responses || ShowSent )
      printf("\n");
    if( Debugging )
      printf("Debugging ON.  ");
    if( Responses )
      printf("SMTP responses to stderr.  ");
    if( ShowSent )
      printf("Sent data to stderr.");
    if( Debugging || Responses || ShowSent )
      printf("\n\n");

    if( CheckComms() ) {
      if( !SendMail() && !Quiet && !Debugging ) {
        printf("Send failed - no message sent.\n");
        Errors = TRUE;
      }
    }
    else if( !Quiet && !Debugging ) {
      printf("Comms check failed - no message sent.\n");
      Errors = TRUE;
    }
  }
  else {
    printf("%s: SMTP mail sender.      v%s      brettski@myrealbox.com\n\n",
           argv[ 0 ], S_VersionString);

    printf("  %s <recipient> <mail host> [options]\n", argv[ 0 ]);
    printf("  -123      : Send only last 123 bytes of file.\n");
    printf("  +123      : Send only first 123 bytes of file.\n");
    printf("  -a        : Disable '<' and '>' around mail addresses.\n");
    printf("  -b        : Send no message body at all.\n");
    printf("  -c        : Suppress To: and From: in mail header.\n");
    printf("  -ddomain  : Domain to put in HELO statement.\n");
    printf("  -e        : Show sent data on stderr.\n");
    printf("  -faddress : From address. Defaults to 'test@test'.\n");
    printf("  -gaddress : Override FROM address in message header.\n");
    printf("  -i        : Don't send date in header.\n");
    printf("  -l        : Send bare LF - not CR LF as per spec.\n");
    printf("  -mfile    : Use file as message text.\n");
    printf("  -n        : Use MIME encoding for message.\n");
    printf("  -p        : Don't report percentage of message file sent.\n");
    printf("  -q        : Quiet mode - print nothing.\n");
    printf("  -r        : Print SMTP responses to stderr.\n");
    printf("  -ssubject : Subject of message. Defaults to 'Test Message'.\n");
    printf("  -taddress : Override TO address in message header.\n");
    printf("  -x        : Delete attached file after successful send.\n");

    printf("\n  -- More --");
    getch();

    printf("\r  -yn       : Close screen after successful send. Optional 'n'\n");
    printf("              will pause for 'n' seconds if errors then close.\n");
    printf("  -z        : Turn on debugging.\n");


    printf("NOTES:       \n");
    printf("   -t defaults to <recipient>.\t");
    printf("-g defaults to -f not specified.\n");
    printf("   -c overrides -f and -t.\t");
    printf("-p, -r and -z override -q.\n");
    printf("   -y defaults to 0 (pause indefinitely).\n");
  }

#ifdef NETWARE
  if( Quiet )
    DestroyScreen(GetCurrentScreen());
  else if( AutoClose ) {
    if( Errors ) {
      printf("\nErrors during send process.\n");
      if( WaitOnErrorTime ) {
        while( WaitOnErrorTime-- && !kbhit() ) {
          printf("\r  Screen will close in %d seconds  ", WaitOnErrorTime);
          sleep(1);
        }
      }
      else {
        printf("  Waiting for key press...");
        getch();
      }
    }
    DestroyScreen(GetCurrentScreen());
  }
#endif
} /* End of main() */

/* End of file: send.c */
